package tools.file;

import java.util.ArrayList;

/**
 * One node of a filetree. This could be a file or a directory.
 * 
 * @author Martin
 * 
 */

public class FileTreeNode {
	/**
	 * name of node
	 */
	private String name = "";

	/**
	 * true, if node is directory
	 */
	private boolean isDirectory = false;

	/**
	 * child nodes
	 */
	private ArrayList<FileTreeNode> childNodes = new ArrayList<FileTreeNode>();

	/**
	 * parent of node. null, if root of tree
	 */
	private FileTreeNode parent;

	/**
	 * get filesystem-path to this node
	 * 
	 * @return
	 */
	public String getPath() {
		if (this instanceof FileTree) {
			FileTree tree = (FileTree) this;
			return tree.getBasePath().toString(false);
		} else {
			String s = parent.getPath() + name;
			if (isDirectory) {
				s += "/";
			}
			return s;
		}
	}

	/**
	 * constructor
	 * 
	 * @param name
	 * @param parent
	 * @param isDirectory
	 */
	public FileTreeNode(String name, FileTreeNode parent, boolean isDirectory) {
		this.name = name;
		this.parent = parent;
		this.isDirectory = isDirectory;
		if (parent != null) {
			parent.addChild(this);
		}

	}

	/**
	 * add child to node
	 * 
	 * @param fileTreeNode
	 *            child-FileTreeNode
	 */
	public void addChild(FileTreeNode fileTreeNode) {
		childNodes.add(fileTreeNode);
		fileTreeNode.parent = this;
	}

	/**
	 * 
	 * @return String Name of FileTreeNode
	 */
	public String getName() {
		return name;
	}

	/**
	 * set name of FileTreeNode
	 * 
	 * @param name
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * 
	 * @return true, if FileTreeNode is directory
	 */
	public boolean isDirectory() {
		return isDirectory;
	}

	/**
	 * set isDirectory
	 * 
	 * @param isDirectory
	 */
	public void setDirectory(boolean isDirectory) {
		this.isDirectory = isDirectory;
	}

	/**
	 * 
	 * @return ArrayList<FileTreeNode> child nodes
	 */
	public ArrayList<FileTreeNode> getChildNodes() {
		return childNodes;
	}

	/**
	 * set ArrayList<FileTreeNode> child nodes
	 * 
	 * @param childNodes
	 *            child nodes
	 */
	public void setChildNodes(ArrayList<FileTreeNode> childNodes) {
		this.childNodes = childNodes;
	}

	/**
	 * get parent node
	 * 
	 * @return
	 */
	public FileTreeNode getParent() {
		return parent;
	}

	/**
	 * set parent node
	 * 
	 * @param parent
	 */
	public void setParent(FileTreeNode parent) {
		this.parent = parent;
	}

	/**
	 * 
	 * @return path relative to root of fileTree
	 */
	public Path getRelativePath() {
		if (this instanceof FileTree) {
			return new Path();
		} else {
			if (isDirectory) {
				Path path = parent.getRelativePath();
				path.add(name);
				return path;
			} else {
				Path path = parent.getRelativePath();
				path.setFilename(name);
				return path;
			}
		}
	}

	/**
	 * 
	 * @return true, if file at node represents a wiki page
	 */
	public boolean isWikiPage() {
		boolean ret = false;
		if (name.length() > 4
				&& name.substring(name.length() - 4, name.length())
						.compareToIgnoreCase(".txt") == 0) {
			ret = true;
		}
		return ret;
	}

	public int getSize() {
		int anz = 0;
		for (FileTreeNode child : childNodes) {
			anz += child.getSize();
		}

		if (!isDirectory && isWikiPage()) {
			anz += 1; // "+ 1" for node itself
		}

		return anz;
	}

}
