package gui.choosedialog;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.WindowConstants;
import javax.swing.border.Border;

/**
 * 
 * @author Martin Pabst, 2009
 * 
 * Simple configurable dialog with radiobuttons and checkboxes.
 * 
 */
public class ChooseDialog extends JDialog implements ActionListener {

	/**
	 * index of radiobutton which is selected when OK-Button is pressed.
	 */
	private int optionChosen = -1;

	/**
	 * Array with all radiobuttons
	 */
	private JRadioButton[] radioButtons;

	/**
	 * Array with all checkboxes
	 */
	private JCheckBox[] checkBoxes;

	/**
	 * selected-state of checkboxes
	 */
	private boolean[] checkBoxState;

	/**
	 * OK-Button
	 */
	private JButton buttonOK = new JButton("OK");

	/**
	 * Cancel-Button
	 */
	private JButton buttonCancel = new JButton("Abbrechen");

	/**
	 * true, if Cancel-button had been pressed
	 */
	private boolean cancelPressed;

	/**
	 * returns true, if cancel-button had been pressed
	 * 
	 * @return
	 */
	public boolean isCancelPressed() {
		return cancelPressed;
	}

	/**
	 * constructor.
	 * 
	 * @param owner
	 *            Main JFrame of program
	 * @param modal
	 *            true, if dialog should be modal relative to owner
	 * @param radioButtonOptions
	 *            String[] with options presented as radiobuttons
	 * @param radioButtonInitialValue
	 *            int index of radiobutton (first button == 0) which is
	 *            initially selected
	 * @param title
	 *            String title of dialog window
	 * @param radioButtonQuestion
	 *            String Question appearing as caption of JPanel which holds
	 *            radiobuttons
	 * @param checkBoxLabels
	 *            String[] labels of checkboxes
	 * @param checkBoxQuestion
	 *            String caption of JPanel which holds checkboxes
	 * @param checkBoxInitialValues
	 *            boolean[] initial selected-states of checkboxes
	 */
	public ChooseDialog(JFrame owner, boolean modal,
			String[] radioButtonOptions, int radioButtonInitialValue,
			String title, String radioButtonQuestion, String[] checkBoxLabels,
			String checkBoxQuestion, boolean[] checkBoxInitialValues) {

		super(owner, modal);

		setSize(200, 200);
		setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

		setTitle(title);

		JPanel mainPanel = new JPanel();
		mainPanel.setLayout(new BorderLayout());
		mainPanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
		setIconImage(owner.getIconImage()); // same icon as owner-window

		getContentPane().setLayout(new BorderLayout());
		getContentPane().add(mainPanel, BorderLayout.CENTER);

		/**
		 * Radiobuttons
		 */
		JPanel radioButtonPanel = new JPanel();
		radioButtonPanel.setLayout(new BoxLayout(radioButtonPanel,
				BoxLayout.Y_AXIS));

		Border blackline = BorderFactory.createLineBorder(Color.black);
		radioButtonPanel.setBorder(BorderFactory.createTitledBorder(blackline,
				radioButtonQuestion));

		ButtonGroup bg = new ButtonGroup();

		radioButtons = new JRadioButton[radioButtonOptions.length];
		for (int i = 0; i < radioButtonOptions.length; i++) {
			JRadioButton rb = new JRadioButton(radioButtonOptions[i]);
			radioButtons[i] = rb;
			radioButtonPanel.add(rb);
			rb.addActionListener(this);
			bg.add(rb);
		}

		radioButtons[radioButtonInitialValue].setSelected(true);

		if (radioButtonOptions.length > 0) {
			mainPanel.add(radioButtonPanel, BorderLayout.NORTH);
		}

		/**
		 * CheckBoxes
		 */
		JPanel checkBoxPanel = new JPanel();
		checkBoxPanel.setLayout(new BoxLayout(checkBoxPanel, BoxLayout.Y_AXIS));

		checkBoxPanel.setBorder(BorderFactory.createTitledBorder(blackline,
				checkBoxQuestion));

		int n = checkBoxLabels.length;
		if (checkBoxInitialValues.length < n) {
			n = checkBoxInitialValues.length;
		}

		checkBoxState = new boolean[n];
		checkBoxes = new JCheckBox[n];

		for (int i = 0; i < n; i++) {
			JCheckBox cb = new JCheckBox(checkBoxLabels[i]);
			cb.setSelected(checkBoxInitialValues[i]);
			checkBoxes[i] = cb;
			checkBoxPanel.add(cb);
		}

		if (n > 0) {
			mainPanel.add(checkBoxPanel, BorderLayout.CENTER);
		}

		/**
		 * Buttons for OK and Cancel
		 */

		JPanel buttonPanel = new JPanel();

		buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.X_AXIS));
		buttonPanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));

		buttonPanel.add(buttonOK);
		buttonPanel.add(Box.createRigidArea(new Dimension(10, 0)));
		buttonPanel.add(buttonCancel);

		buttonOK.addActionListener(this);
		buttonCancel.addActionListener(this);

		mainPanel.add(buttonPanel, BorderLayout.SOUTH);

		optionChosen = -1;

	}

	/**
	 * action delegator for OK-Button, Cancel-Button and checkboxes
	 */
	@Override
	public void actionPerformed(ActionEvent e) {
		if (e.getSource() == buttonCancel) {
			optionChosen = -1;
			setVisible(false);
		}
		if (e.getSource() == buttonOK) {
			for (int i = 0; i < radioButtons.length; i++) {
				if (radioButtons[i].isSelected()) {
					optionChosen = i;
				}
			}

			for (int i = 0; i < checkBoxes.length; i++) {
				checkBoxState[i] = checkBoxes[i].isSelected();
			}

			setVisible(false); // closes dialog and unblocks caller of
			// setvisible(true)
		}
	}

	/**
	 * get index of radiobutton (first == 0) which is selected. Returns -1 if
	 * Cancel-Button had been pressed.
	 * 
	 * @return
	 */
	public int getOptionChosen() {
		return optionChosen;
	}

	/**
	 * get selected-state of checkboxes
	 * 
	 * @return
	 */
	public boolean[] getCheckBoxState() {
		return checkBoxState;
	}

}
